<?php

namespace App;

use DB;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Category extends Model
{
    use SoftDeletes;

    /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * Combines categories into a recursive tree.
     *
     * @param  int  $business_id
     * @return array
     */
    public static function catAndSubCategories($business_id)
    {
        $all_categories = Category::where('business_id', $business_id)
                                ->where('category_type', 'product')
                                ->orderBy('name', 'asc')
                                ->get()
                                ->toArray();

        if (empty($all_categories)) {
            return [];
        }

        // Start at level 0
        return self::buildCategoryTree($all_categories);
    }

    /**
     * Recursively build a tree of categories.
     *
     * @param  array  $categories  A flat array of categories
     * @param  int  $parent_id  The starting parent id (default: 0 for top-level)
     * @param  int  $level  The current level (default: 0)
     * @return array
     */
    protected static function buildCategoryTree($categories, $parent_id = 0, $level = 0)
    {
        $branch = [];
        foreach ($categories as $category) {
            if ($category['parent_id'] == $parent_id) {
                // Recursively find children
                $children = self::buildCategoryTree($categories, $category['id'], $level + 1);
                if ($children) {
                    if ($level == 0) {
                        // Direct children (sub-categories)
                        $category['sub_categories'] = $children;
                    } elseif ($level == 1) {
                        // Second-level children (sub–sub–categories)
                        $category['sub_sub_categories'] = $children;
                    } else {
                        // For levels deeper than 1, you can decide whether to keep nesting under sub_sub_categories
                        $category['sub_sub_categories'] = $children;
                    }
                }
                $branch[] = $category;
            }
        }
        return $branch;
    }

    /**
     * Relationship to get immediate sub-categories.
     */
    public function sub_categories()
    {
        return $this->hasMany(\App\Category::class, 'parent_id');
    }

    /**
     * Scope a query to only include main categories.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeOnlyParent($query)
    {
        return $query->where('parent_id', 0);
    }

    /**
     * For Dropdown.
     *
     * @param  int  $business_id
     * @param  string  $type  Category type
     * @return \Illuminate\Support\Collection
     */
    public static function forDropdown($business_id, $type)
    {
        $categories = Category::where('business_id', $business_id)
                            ->where('category_type', $type)
                            ->select(DB::raw('IF(short_code IS NOT NULL, CONCAT(name, "-", short_code), name) as name'), 'id')
                            ->orderBy('name', 'asc')
                            ->get();

        return $categories->pluck('name', 'id');
    }
}
