<?php

namespace App\Http\Controllers;

use App\Category;
use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class TaxonomyController extends Controller
{
    /**
     * All Utils instance.
     */
    protected $moduleUtil;

    /**
     * Constructor
     *
     * @param  ModuleUtil  $moduleUtil
     * @return void
     */
    public function __construct(ModuleUtil $moduleUtil)
    {
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $category_type = request()->get('type');
        if ($category_type == 'product' && ! auth()->user()->can('category.view') && ! auth()->user()->can('category.create')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $can_edit = true;
            if ($category_type == 'product' && ! auth()->user()->can('category.update')) {
                $can_edit = false;
            }

            $can_delete = true;
            if ($category_type == 'product' && ! auth()->user()->can('category.delete')) {
                $can_delete = false;
            }

            $business_id = request()->session()->get('user.business_id');

            $category = Category::where('business_id', $business_id)
                            ->where('category_type', $category_type)
                            ->select(['name', 'short_code', 'description', 'id', 'parent_id']);

            return Datatables::of($category)
                ->addColumn(
                    'action', function ($row) use ($can_edit, $can_delete, $category_type) {
                        $html = '';
                        if ($can_edit) {
                            $html .= '<button data-href="'.action([\App\Http\Controllers\TaxonomyController::class, 'edit'], [$row->id]).'?type='.$category_type.'" class="btn btn-xs btn-primary edit_category_button"><i class="glyphicon glyphicon-edit"></i>'.__('messages.edit').'</button>';
                        }

                        if ($can_delete) {
                            $html .= '&nbsp;<button data-href="'.action([\App\Http\Controllers\TaxonomyController::class, 'destroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_category_button"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                        }

                        return $html;
                    }
                )
                ->editColumn('name', function ($row) {
                    if ($row->parent_id != 0) {
                        return '--'.$row->name;
                    } else {
                        return $row->name;
                    }
                })
                ->removeColumn('id')
                ->removeColumn('parent_id')
                ->rawColumns(['action'])
                ->make(true);
        }

        $module_category_data = $this->moduleUtil->getTaxonomyData($category_type);

        return view('taxonomy.index')->with(compact('module_category_data', 'module_category_data'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * When creating a taxonomy, if the type is "sub_category" then the parent
     * must be chosen from the main categories. If the type is "sub_sub_category"
     * then the parent must be chosen from the sub categories.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $category_type = request()->get('type');
        if ($category_type == 'product' && ! auth()->user()->can('category.create')) {
            abort(403, 'Unauthorized action.');
        }
        $business_id = request()->session()->get('user.business_id');

        $module_category_data = $this->moduleUtil->getTaxonomyData($category_type);

        // Determine the available parent categories based on taxonomy type.
        $parent_categories = [];
        if ($category_type == 'sub_sub_category') {
            // Parent for sub–sub category must be a sub category.
            $parent_categories = Category::where('business_id', $business_id)
                                  ->where('parent_id', '<>', 0)
                                  ->where('category_type', 'sub_category')
                                  ->pluck('name', 'id')
                                  ->toArray();
        } elseif ($category_type == 'sub_category') {
            // Parent for sub category must be a main category.
            $parent_categories = Category::where('business_id', $business_id)
                                  ->where('parent_id', 0)
                                  ->where('category_type', 'category')
                                  ->pluck('name', 'id')
                                  ->toArray();
        }
        // If taxonomy type is "category", no parent is needed.

        // Set flag for view usage.
        $is_sub_sub = ($category_type == 'sub_sub_category') ? true : false;

        return view('taxonomy.create')
                    ->with(compact('parent_categories', 'module_category_data', 'category_type', 'is_sub_sub'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * If adding a sub–sub taxonomy, the parent is taken from the "sub_parent_id"
     * field; if adding a sub taxonomy, from the "parent_id" field.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $category_type = $request->input('category_type');
        if ($category_type == 'product' && ! auth()->user()->can('category.create')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $input = $request->only(['name', 'short_code', 'category_type', 'description']);
            // For sub–sub taxonomy, check for the sub_parent_id first.
            if (! empty($request->input('add_as_sub_sub_cat')) && $request->input('add_as_sub_sub_cat') == 1 && ! empty($request->input('sub_parent_id'))) {
                $input['parent_id'] = $request->input('sub_parent_id');
            } elseif (! empty($request->input('add_as_sub_cat')) && $request->input('add_as_sub_cat') == 1 && ! empty($request->input('parent_id'))) {
                $input['parent_id'] = $request->input('parent_id');
            } else {
                $input['parent_id'] = 0;
            }
            $input['business_id'] = $request->session()->get('user.business_id');
            $input['created_by'] = $request->session()->get('user.id');

            $category = Category::create($input);
            $output = ['success' => true,
                'data' => $category,
                'msg' => __('category.added_success'),
            ];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().' Line:'.$e->getLine().' Message:'.$e->getMessage());

            $output = ['success' => false,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return $output;
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Category  $category
     * @return \Illuminate\Http\Response
     */
    public function show(Category $category)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * For editing, if the taxonomy type is "sub_sub_category" then the parent should
     * be chosen from sub categories; if "sub_category", from main categories.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $category_type = request()->get('type');
        if ($category_type == 'product' && ! auth()->user()->can('category.update')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');
            $category = Category::where('business_id', $business_id)->find($id);

            $module_category_data = $this->moduleUtil->getTaxonomyData($category_type);

            // Adjust available parent categories based on taxonomy type.
            if ($category_type == 'sub_sub_category') {
                // Parent must be a sub category.
                $parent_categories = Category::where('business_id', $business_id)
                                            ->where('parent_id', '<>', 0)
                                            ->where('category_type', 'sub_category')
                                            ->where('id', '!=', $id)
                                            ->pluck('name', 'id');
            } elseif ($category_type == 'sub_category') {
                // Parent must be a main category.
                $parent_categories = Category::where('business_id', $business_id)
                                            ->where('parent_id', 0)
                                            ->where('category_type', 'category')
                                            ->where('id', '!=', $id)
                                            ->pluck('name', 'id');
            } else {
                $parent_categories = [];
            }
            
            $is_parent = false;
            if ($category->parent_id == 0) {
                $is_parent = true;
                $selected_parent = null;
            } else {
                $selected_parent = $category->parent_id;
            }

            return view('taxonomy.edit')
                ->with(compact('category', 'parent_categories', 'is_parent', 'selected_parent', 'module_category_data'));
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * When updating, if the taxonomy is being set as a sub–sub taxonomy, the
     * parent_id is taken from sub_parent_id; otherwise from parent_id.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        if (request()->ajax()) {
            try {
                $input = $request->only(['name', 'description']);
                $business_id = $request->session()->get('user.business_id');

                $category = Category::where('business_id', $business_id)->findOrFail($id);

                if ($category->category_type == 'product' && ! auth()->user()->can('category.update')) {
                    abort(403, 'Unauthorized action.');
                }

                $category->name = $input['name'];
                $category->description = $input['description'];
                $category->short_code = $request->input('short_code');

                if (! empty($request->input('add_as_sub_sub_cat')) && $request->input('add_as_sub_sub_cat') == 1 && ! empty($request->input('sub_parent_id'))) {
                    $category->parent_id = $request->input('sub_parent_id');
                } elseif (! empty($request->input('add_as_sub_cat')) && $request->input('add_as_sub_cat') == 1 && ! empty($request->input('parent_id'))) {
                    $category->parent_id = $request->input('parent_id');
                } else {
                    $category->parent_id = 0;
                }
                $category->save();

                $output = ['success' => true,
                    'msg' => __('category.updated_success'),
                ];
            } catch (\Exception $e) {
                \Log::emergency('File:'.$e->getFile().' Line:'.$e->getLine().' Message:'.$e->getMessage());

                $output = ['success' => false,
                    'msg' => __('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if (request()->ajax()) {
            try {
                $business_id = request()->session()->get('user.business_id');

                $category = Category::where('business_id', $business_id)->findOrFail($id);

                if ($category->category_type == 'product' && ! auth()->user()->can('category.delete')) {
                    abort(403, 'Unauthorized action.');
                }

                $category->delete();

                $output = ['success' => true,
                    'msg' => __('category.deleted_success'),
                ];
            } catch (\Exception $e) {
                \Log::emergency('File:'.$e->getFile().' Line:'.$e->getLine().' Message:'.$e->getMessage());

                $output = ['success' => false,
                    'msg' => __('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    public function getCategoriesApi()
    {
        try {
            $api_token = request()->header('API-TOKEN');

            $api_settings = $this->moduleUtil->getApiSettings($api_token);

            $categories = Category::catAndSubCategories($api_settings->business_id);
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().' Line:'.$e->getLine().' Message:'.$e->getMessage());

            return $this->respondWentWrong($e);
        }

        return $this->respond($categories);
    }

    /**
     * get taxonomy index page
     * through ajax
     *
     * @return \Illuminate\Http\Response
     */
    public function getTaxonomyIndexPage(Request $request)
    {
        if (request()->ajax()) {
            $category_type = $request->get('category_type');
            $module_category_data = $this->moduleUtil->getTaxonomyData($category_type);

            return view('taxonomy.ajax_index')
                ->with(compact('module_category_data', 'category_type'));
        }
    }

    /**
     * Get sub-categories (i.e. child taxonomies) for a given parent.
     * This method is used via AJAX to populate the sub–sub category dropdown.
     *
     * Expects a POST request with:
     * - parent_id: The ID of the taxonomy whose children are to be returned.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getSubCategories(Request $request)
    {
        if (! empty($request->input('parent_id'))) {
            $parent_id = $request->input('parent_id');
            $business_id = $request->session()->get('user.business_id');

            // Retrieve the full nested taxonomy tree for this business.
            $all_categories = Category::catAndSubCategories($business_id);

            // Recursively search the tree for the node with the given ID.
            $findNode = function($categories, $id) use (&$findNode) {
                foreach ($categories as $cat) {
                    if ($cat['id'] == $id) {
                        return $cat;
                    }
                    if (! empty($cat['sub_categories'])) {
                        $found = $findNode($cat['sub_categories'], $id);
                        if ($found) {
                            return $found;
                        }
                    }
                }
                return null;
            };

            $node = $findNode($all_categories, $parent_id);

            $html = '<option value="">None</option>';
            if ($node && ! empty($node['sub_categories'])) {
                // Flatten the nested children with indentation.
                $flatten = function($categories, $level = 0) use (&$flatten) {
                    $result = [];
                    foreach ($categories as $cat) {
                        $cat['level'] = $level;
                        $result[] = $cat;
                        if (! empty($cat['sub_categories'])) {
                            $result = array_merge($result, $flatten($cat['sub_categories'], $level + 1));
                        }
                    }
                    return $result;
                };

                $flattened = $flatten($node['sub_categories'], 0);
                foreach ($flattened as $child) {
                    $indent = str_repeat('-- ', $child['level']);
                    $html .= '<option value="' . $child['id'] . '">' . $indent . $child['name'] . '</option>';
                }
            }
            return response()->json($html);
        }
    }
}
